﻿///
/// 格納アイテムの階層構造を表現するためのコンテナクラスのヘッダファイル。
///

#if       !defined(INCLUDED_SZ_STORED_ITEM_CONTAINER)
#define            INCLUDED_SZ_STORED_ITEM_CONTAINER

#if       !defined(INCLUDED_SZ_COMMON)
#include  <szCommon.hpp>
#endif // !defined(INCLUDED_SZ_COMMON)

#if       !defined(INCLUDED_SZ_TIME)
#include  <szTime.hpp>
#endif // !defined(INCLUDED_SZ_TIME)

#if       !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)
#include  <szForwardDeclarations.hpp>
#endif // !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)

SZ_NS_BEG(szpp)

/// アーカイブ内の格納アイテムを保持するコンテナを表現するクラス。
class SZ_SPEC StoredItemContainer : boost::noncopyable
{
public:

  /// コンテナの名称を設定するコンストラクタ。
  StoredItemContainer(const szstring & name);
  /// デストラクタ。
  ~StoredItemContainer();

  //  アイテムへのアクセッサ

  /// コンテナの名称を取得するメソッド。
  const szstring &GetName() const { return name; }
  /// コンテナに対応する格納アイテム（もしあれば）のポインタを取得するメソッド。
  StoredItem     *GetItem() { return thisItem; }
  /// コンテナに対応する格納アイテムのポインタを設定するメソッド。
  void            SetItem(StoredItem *item);

  /// コンテナのサブアイテムの個数を取得するメソッド。
  index_t           GetNumberOfSubItems() const { return subItems.size(); }
  /// コンテナの特定のサブアイテムを取得するメソッド。
  StoredItem       *GetSubItem(index_t index) { return subItems[index]; }
  /// コンテナの特定のサブアイテムを取得するメソッド。
  const StoredItem *GetSubItem(index_t index) const { return subItems[index]; }
  /// コンテナのサブアイテムを名称で検索するメソッド。
  StoredItem       *FindSubItem(const szstring &targetName);
  /// コンテナのサブアイテムを名称で検索するメソッド。
  const StoredItem *FindSubItem(const szstring &targetName) const;
  /// コンテナのサブアイテムを追加するメソッド。
  void              AddSubItem(StoredItem *item);

  //  サブコンテナへのアクセッサ

  /// コンテナのサブコンテナの個数を取得するメソッド。
  index_t                     GetNumberOfSubContainers() const { return subContainers.size(); }
  /// コンテナの特定のサブコンテナを取得するメソッド。
  StoredItemContainer *       GetSubContainer(index_t index) { return subContainers[index]; }
  /// コンテナの特定のサブコンテナを取得するメソッド。
  const StoredItemContainer * GetSubContainer(index_t index) const { return subContainers[index]; }
  /// コンテナのサブコンテナを名称で検索するメソッド。
  StoredItemContainer *       FindSubContainer(const szstring &targetName);
  /// コンテナのサブコンテナを名称で検索するメソッド。
  const StoredItemContainer * FindSubContainer(const szstring &targetName) const;
  /// コンテナの子孫コンテナ（コンテナのみ）を名称で検索するメソッド。
  StoredItemContainer *       FindDescendant(const szstring &targetName);
  /// コンテナの子孫コンテナ（コンテナのみ）を名称で検索するメソッド。
  const StoredItemContainer * FindDescendant(const szstring &targetName) const;
  /// コンテナのサブコンテナを追加するメソッド。
  void                        AddSubContainer(StoredItemContainer *subcontainer);

  ///  親コンテナのポインタを取得するメソッド。
  StoredItemContainer * GetParent() { return parent; }
  ///  親コンテナのポインタを設定するメソッド。
  void                  SetParent(StoredItemContainer * p) { parent = p; }

private:

  /// コンテナ自身に相当する格納アイテムへのポインタ（所有権なし）。該当する格納アイテムがない場合は NULL。
  StoredItem *thisItem;

  /// <summary>
  /// コンテナに含まれる格納アイテム群のポインタ（所有権なし）のベクタ。
  /// </summary>
  /// <remarks>
  /// 終端ノードである格納アイテムのみが本メンバで管理されるので、ファイルを列挙する場合には subItems を走査すればよい。<p>
  /// 仮に tar アーカイブに aDir という名前のディレクトリ格納アイテムが含まれていた場合、
  /// それは subItems ではなく subContainers の方で管理される。
  /// 具体的には aDir という名称を持つサブコンテナ aContainer が一つ存在し、
  /// aContainer->thisItem が aDir の情報を保持するディレクトリ格納アイテムを指すことになる。
  /// </remarks>
  std::vector<StoredItem *> subItems;

  /// <summary>
  /// コンテナに含まれる格納アイテムコンテナ群のポインタ（所有権あり）のベクタ。
  /// </summary>
  /// <remarks>
  /// サブフォルダを列挙する場合には subContainers を走査すればよい。
  /// </remarks>
  std::vector<StoredItemContainer *> subContainers;

  /// <summary>
  /// 親コンテナのポインタ（所有権なし）。
  /// NULL の場合はルートコンテナであることを意味する。
  /// </summary>
  StoredItemContainer *parent;

  /// <summary>
  /// 上位コンテナ内における正規化済み相対パス文字列（要するにフォルダ名）。
  /// </summary>
  /// <remarks>
  /// 文字列の末尾にバックスラッシュは<b>付けない</b>。
  /// また、ルートコンテナの（本来は空文字列のはずの）パス文字列にアーカイブ名が設定されていることがある。
  /// これにより eo 相当のディレクトリ自動作成機能が実現することが可能になる。
  /// </remarks>
  /// <notes>
  /// name と thisItem->name が一致しないこともある。
  /// 例えば tar アーカイブで文字ケースのみが異なるファイルとディレクトリが存在した場合、後で出現した方が自動的に改名される。
  /// 通常はディレクトリが先に出現するので、ディレクトリが改名されるケースは想定しがたいが、理論上はディレクトリも改名されうる。<p>
  /// なお、階層構造を構築する段階では自動改名をしてはいけない。
  /// なぜなら、その後に出現する当該ディレクトリに属する格納ファイルのパスと一致しなくなるため、
  /// マッチングが出来なくなってしまうからである。
  /// 大文字小文字を区別せずに階層構造を構築した後、別フェーズで改名処理を行うこと。
  /// <notes>
  szstring name;
};

SZ_NS_END(szpp)

#endif // !defined(INCLUDED_SZ_STORED_ITEM_CONTAINER)
